package com.tencent.start.cgs.tools;

import org.apache.commons.codec.binary.Hex;

import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Random;
import java.util.zip.CRC32;

@SuppressWarnings("unused")
public class App {
    public static final String VERSION = BuildConfig.VERSION + "."
            + BuildConfig.BUILD_NO + " (" + BuildConfig.BUILD_TIME + ")";
    public static boolean ENABLE_DEBUG = false;
    public static final long SIZE_4G = 1024L * 1024L * 1024L * 4;
    static final SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static final Random rand = new SecureRandom(String.valueOf(System.currentTimeMillis()).getBytes());

    public static File getCurrentDirectory() throws IOException {
        return new File(new File(".").getCanonicalPath());
    }

    public static boolean isTextEmpty(String s) {
        return null == s || 0 == s.length();
    }

    public static String addBackslash(String path) {
        return path.isEmpty() || path.endsWith("/") ? path : path + "/";
    }

    public static int javaToDosTime(long time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(time);
        int year = calendar.get(Calendar.YEAR) - 1980;
        if (year < 0) {
            return (1 << 21) | (1 << 16);
        }
        return (int) ((year << 25 |
                (calendar.get(Calendar.MONTH) + 1) << 21 |
                calendar.get(Calendar.DAY_OF_MONTH) << 16 |
                calendar.get(Calendar.HOUR_OF_DAY) << 11 |
                calendar.get(Calendar.MINUTE) << 5 |
                calendar.get(Calendar.SECOND) >> 1) & 0xffffffffL);
    }

    @SuppressWarnings("all")
    public static long dosToJavaTime(long dtime) {
        int year = (int)(((dtime >> 25) & 0x7f) + 1980);
        int month =  (int)(((dtime >> 21) & 0x0f) - 1);
        int day = (int)((dtime >> 16) & 0x1f);
        int hour = (int)((dtime >> 11) & 0x1f);
        int minute = (int)((dtime >> 5) & 0x3f);
        int second = (int)((dtime << 1) & 0x3e);
        Calendar calendar = Calendar.getInstance();
        calendar.set(year, month, day, hour, minute, second);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTimeInMillis();
    }

    public static void Assert(boolean condition) {
        Assert(condition, "");
    }

    public static void Assert(boolean condition, String msg) {
        if (!condition) {
            throw new AssertionError(msg);
        }
    }

    public static String humanReadableBytes(long bytes) {
        if (bytes >= 1099511627776L) {
            return String.format("%.2f", (double) bytes / 1099511627776L) + "T";
        } else if (bytes >= 1073741824L) {
            return String.format("%.2f", (double) bytes / 1073741824L) + "G";
        } else if (bytes >= 1048576L) {
            return String.format("%.2f", (double) bytes / 1048576L) + "M";
        } else if (bytes >= 1024L) {
            return String.format("%.2f", (double) bytes / 1024L) + "K";
        } else {
            return bytes + "B";
        }
    }

    public static String humanReadableBytes(double bytes) {
        if (Double.compare(bytes, 1099511627776.0) >= 0) {
            return String.format("%.2f", bytes / 1099511627776.0) + "T";
        } else if (Double.compare(bytes, 1073741824.0) >= 0) {
            return String.format("%.2f", bytes / 1073741824.0) + "G";
        } else if (Double.compare(bytes, 1048576.0) >= 0) {
            return String.format("%.2f", bytes / 1048576.0) + "M";
        } else if (Double.compare(bytes, 1024.0) >= 0) {
            return String.format("%.2f", bytes / 1024.0) + "K";
        } else {
            return ((long) bytes) + "B";
        }
    }

    public static String humanReadableTime(long ms) {
        if (ms < 1000) {
            return String.format("%dms", (int) ms);
        }
        boolean printZero = false;
        StringBuilder sb = new StringBuilder(24);
        if (ms >= 86400000) {
            sb.append(String.format("%dd", (int) (ms / 86400000)));
            ms = ms % 86400000;
            printZero = true;
        }
        if (ms >= 3600000) {
            sb.append(String.format("%dh", (int) (ms / 3600000)));
            ms = ms % 3600000;
            printZero = true;
        } else if (printZero) {
            sb.append("0h");
        }
        if (ms >= 60000) {
            sb.append(String.format("%dm", (int) (ms / 60000)));
            ms = ms % 60000;
        } else if (printZero) {
            sb.append("0m");
        }
        sb.append(String.format("%ds", (int) (ms / 1000)));
        return sb.toString();
    }

    public static String humanReadableTimeNanoSeconds(long nanoseconds) {
        if (nanoseconds < 1000L) {
            return String.format("%dns", (int) nanoseconds);
        } else if (nanoseconds < 1000000000L) {
            return String.format("%.3fms", (double) nanoseconds / 1000000.0);
        }
        return humanReadableTime(nanoseconds / 1000000L);
    }

    public static class Speedometer {
        private long bytes = 0;

        private final long totalBytes;

        private final String totalBytesString;

        private final long startTimeNanoSeconds;

        private long nextTraceTimeNanoSeconds;

        private final long traceIntervalNanoSeconds;

        public Speedometer(long totalBytes, long traceIntervalNanoSeconds) {
            this.totalBytes = totalBytes;
            this.totalBytesString = humanReadableBytes(totalBytes);
            this.startTimeNanoSeconds = System.nanoTime();
            this.traceIntervalNanoSeconds = traceIntervalNanoSeconds;
            this.nextTraceTimeNanoSeconds = this.startTimeNanoSeconds + traceIntervalNanoSeconds;
        }

        public Speedometer(long totalBytes) {
            this(totalBytes, 2000000000L);
        }

        public void feed(long num) {
            this.bytes += num;
            final long currentTimeNanoSeconds = System.nanoTime();
            if (this.bytes < totalBytes && currentTimeNanoSeconds >= nextTraceTimeNanoSeconds) {
                trace(currentTimeNanoSeconds);
            }
        }

        public void trace(long currentTimeNanoSeconds) {
            nextTraceTimeNanoSeconds = currentTimeNanoSeconds + traceIntervalNanoSeconds;
            double speedMs = (double) bytes * 1000000.0 / (currentTimeNanoSeconds - startTimeNanoSeconds);
            System.out.println(humanReadableBytes(bytes) + "/" + totalBytesString + ", " +
                    String.format("%.2f", 100.0 * bytes / totalBytes) + '%' + ", " +
                    humanReadableBytes(speedMs * 1000) + "/s, " +
                    humanReadableTime((long) ((double) (totalBytes - bytes) / speedMs)) + " remain.");
        }

        public void finish() {
            final long totalTimeSpendNanoSeconds = System.nanoTime() - startTimeNanoSeconds;
            final double speedMs = (double) bytes * 1000000.0 / totalTimeSpendNanoSeconds;
            System.out.println(totalBytesString + ", " + humanReadableBytes(speedMs * 1000) + "/s, "
                    + humanReadableTimeNanoSeconds(totalTimeSpendNanoSeconds) + " total");
        }

        public long getTotalBytes() {
            return totalBytes;
        }

        public long getBytes() {
            return bytes;
        }

        public long getRemainBytes() {
            return totalBytes - bytes;
        }
    }

    public static String getFileExtension(String name) {
        int n = name.lastIndexOf('.');
        return -1 != n ? name.substring(n) : "";
    }

    public static String getFileExtension(File file) {
        return getFileExtension(file.getName());
    }

    public static long safeParseLong(String s, long def) {
        try {
            return Long.parseLong(s);
        } catch (Exception e) {
            return def;
        }
    }

    public static long calcFileCrc32(AbstractFile file) throws IOException {
        long totalBytesRead = 0;
        CRC32 crc32 = new CRC32();
        try (InputStream in = file.getInputStream(0)) {
            int bytesRead;
            byte[] buffer = new byte[4 * 1024 * 1024];
            while ((bytesRead = in.read(buffer)) > 0) {
                crc32.update(buffer, 0, bytesRead);
                totalBytesRead += bytesRead;
            }
        }
        if (totalBytesRead != file.length()) {
            throw new IOException("totalBytesRead != file.length()");
        }
        return crc32.getValue();
    }

    public static byte[] calcFileMessageDigest(AbstractFile file, String algorithm) throws IOException {
        long totalBytesRead = 0;
        try (InputStream in = file.getInputStream(0)) {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            int bytesRead;
            byte[] buffer = new byte[4 * 1024 * 1024];
            while ((bytesRead = in.read(buffer)) > 0) {
                md.update(buffer, 0, bytesRead);
                totalBytesRead += bytesRead;
            }
            if (totalBytesRead != file.length()) {
                throw new IOException("totalBytesRead != file.length()");
            }
            return md.digest();
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }
    }

    public static String calcFileMessageDigestHexString(AbstractFile file, String algorithm) throws IOException {
        return Hex.encodeHexString(calcFileMessageDigest(file, algorithm));
    }

    public interface Checksum {
        void update(byte[] buf, int off, int len);
    }

    public interface ReadCallback {
        byte[] onRead(byte[] buf, int off, int len) throws IOException;
    }

    @SuppressWarnings("UnusedReturnValue")
    public static long writeFileWithProgress(DataOutput output, AbstractFile file, Checksum checksum)
            throws IOException {
        Speedometer speedometer = new Speedometer(file.length());
        try (InputStream in = file.getInputStream(0)) {
            int bytesRead;
            byte[] buf = new byte[4 * 1024 * 1024];
            while ((bytesRead = in.read(buf)) > 0) {
                output.write(buf, 0, bytesRead);
                if (checksum != null) {
                    checksum.update(buf, 0, bytesRead);
                }
                speedometer.feed(bytesRead);
            }
        }
        if (0 != speedometer.getRemainBytes()) {
            throw new IOException("totalBytesRead != file.length()");
        }
        speedometer.finish();
        return speedometer.getTotalBytes();
    }

    @SuppressWarnings("UnusedReturnValue")
    public static long readFileWithProgress(AbstractFile file, byte[] buf, ReadCallback cb)
            throws IOException {
        Speedometer speedometer = new Speedometer(file.length());
        try (InputStream in = file.getInputStream(0)) {
            int bytesRead;
            while ((bytesRead = in.read(buf = (buf == null ? new byte[4 * 1024 * 1024] : buf))) > 0) {
                buf = cb.onRead(buf, 0, bytesRead);
                speedometer.feed(bytesRead);
            }
        }
        if (0 != speedometer.getRemainBytes()) {
            throw new IOException("totalBytesRead != file.length()");
        }
        speedometer.finish();
        return speedometer.getTotalBytes();
    }

    @SuppressWarnings("UnusedReturnValue")
    public static long readFileWithProgress(AbstractFile file, ReadCallback cb) throws IOException {
        return readFileWithProgress(file, null, cb);
    }

    public static void usage() {
        System.out.println("CGS Game Packet Tools " + VERSION);
        System.out.println("Usage:");
        ZpkFilePack.printUsage();
        ZpkFileUnpack.printUsage();
        ZpkFileDiff.printUsage();
        UtilTools.printUsage();
        System.exit(1);
    }

    public static String randomString(int length) {
        String s = "0123456789abcdefghijklmnopqrstuvwxyz";
        StringBuilder sb = new StringBuilder(length);
        for (int i = 0; i < length; ++i) {
            sb.append(s.charAt(rand.nextInt(s.length())));
        }
        return sb.toString();
    }

    public static String pathJoin(String... s) {
        if (0 == s.length) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        String pre = s[0];
        sb.append(pre);
        for (int i = 1; i < s.length; ++i) {
            String cur = s[i];
            if (pre.endsWith("/")) {
                if (cur.startsWith("/")) {
                    sb.append(cur.substring(1));
                } else {
                    sb.append(cur);
                }
            } else if (cur.startsWith("/")) {
                sb.append(cur);
            } else {
                sb.append('/');
                sb.append(cur);
            }
            pre = cur;
        }
        return sb.toString();
    }

    public static void nop() {}

    public static void nop(Object... objects) {
    }

    public static void UNUSED(Object... objects) {
    }

    private static void callMain(String[] args) throws Exception {
        if (args.length < 3) {
            if (args.length > 0 && args[0].equals("version")) {
                System.out.println(BuildConfig.VERSION + "." + BuildConfig.BUILD_NO);
                return;
            }
            usage();
            return;
        }
        if (args[0].equals(ZpkFile.EXT_NAME.substring(1))) {
            if (args[1].equals("unpack")) {
                ZpkFileUnpack.callMain(args);
            } else if (args[1].equals("diff") ||
                    args[1].equals("patch") ||
                    args[1].equals("update")) {
                ZpkFileDiff.callMain(args);
            } else {
                ZpkFilePack.callMain(args);
            }
            return;
        } else if (args[0].equals("util")) {
            UtilTools.callMain(args);
            return;
        }
        usage();
    }

    public static void main(String[] args) {
        try {
            App.callMain(args);
            System.exit(0);
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(-1);
        }
    }
}
